# ![vode-logo](https://raw.githubusercontent.com/ryupold/vode/refs/heads/main/logo.webp)
| [![TypeScript](https://img.shields.io/badge/TypeScript-100%25-blue?logo=typescript)](https://www.typescriptlang.org/) |  [![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](./LICENSE)   |
| :-------------------------------------------------------------------------------------------------------------------: | :----------------------------------------------------------------------------------: |
|         [![NPM](https://badge.fury.io/js/%40ryupold%2Fvode.svg)](https://www.npmjs.com/package/@ryupold/vode)         | [![Dependencies](https://img.shields.io/badge/dependencies-0-success)](package.json) |

---

A compact web framework for minimalist developers. Zero dependencies, no build step except for typescript compilation, and a simple virtual DOM implementation that is easy to understand and use. Autocompletion out of the box thanks to `lib.dom.d.ts`.

It brings a primitive building block to the table that gives flexibility in composition and makes refactoring easy. 
The use cases can be single page applications or isolated components with complex state.

## Usage

### ESM

```html
<!DOCTYPE html>
<html>
<head>
    <title>ESM Example</title>
</head>
<body>
    <div id="app"></div>
    <script type="module">
        import { app, BR, DIV, INPUT, SPAN } from 'https://unpkg.com/@ryupold/vode/dist/vode.min.mjs';

        const appNode = document.getElementById('app');

        const state = { counter: 0 };

        app(appNode, state,
            (s) => [DIV,
                [INPUT, {
                    type: 'button',
                    onclick: { counter: s.counter + 1 },
                    value: 'Click me',
                }],
                [BR],
                [SPAN, { style: { color: 'red' } }, `${s.counter}`],
            ]
        );
    </script>
</body>
</html>
```

### Classic (iife)
Binds the library to the global `V` variable.

```html
<!DOCTYPE html>
<html>
<head>
    <title>Classic Script Example</title>
    <script src="https://unpkg.com/@ryupold/vode/dist/vode.min.js"></script>
</head>
<body>
    <div id="app"></div>
    <script>
        var appNode = document.getElementById('app');
        
        var state = { counter: 0 };

        V.app(appNode, state,
            (s) => ["div",
                ["input", {
                    type: 'button',
                    onclick: { counter: s.counter + 1 },
                    value: 'Click me',
                }
                ],
                ["br"],
                ["span", { style: { color: 'red' } }, `${s.counter}`],
            ]);
    </script>
</body>
</html>
```

### NPM

[![NPM](https://nodei.co/npm/@ryupold/vode.svg?color=red&data=n,v,s,d,u)](https://www.npmjs.com/package/@ryupold/vode)

index.html

```html
<html>
<head>
    <title>Vode Example</title>
    <script type="module" src="main.js"></script>
</head>
<body>
    <div id="app"></div>
</body>
</html>
```

main.ts
```typescript
import { app, createState, BR, DIV, INPUT, SPAN } from '@ryupold/vode';

const state = createState({
    counter: 0,
});

type State = typeof state;

const appNode = document.getElementById('app')!;

app<State>(appNode, state,
    (s: State) => [DIV,
        [INPUT, {
            type: 'button',
            onclick: { counter: s.counter + 1 },
            value: 'Click me',
        }],
        [BR],
        [SPAN, { style: { color: 'red' } }, `${s.counter}`],
    ]
);
```

## `[V,{},d,e]`

Lets describe UI as data structures that map 1:1 to DOM elements.

A `vode` is a representation of a virtual DOM node, which is a tree structure of HTML elements. It is written as tuple:

```
[TAG, PROPS?, CHILDREN...]
```

As you can see, it is a simple array with the first element being the tag name, the second element being an optional properties object, and the rest being child-vodes.

They are lightweight structures to describe what the DOM should look like.

Imagine this HTML:

```html
<div class="card">
  <div class="card-image">
    <figure class="image is-4by3">
      <img
        src="placeholders/1280x960.png"
        alt="Placeholder image"
      />
    </figure>
  </div>
  <div class="card-content">
    <div class="media">
      <div class="media-left">
        <figure class="image is-48x48">
          <img
            src="placeholders/96x96.png"
            alt="Placeholder image"
          />
        </figure>
      </div>
      <div class="media-content">
        <p class="title is-4">John Smith</p>
        <p class="subtitle is-6">@johnsmith</p>
      </div>
    </div>

    <div class="content">
      Lorem ipsum dolor sit amet, consectetur adipiscing elit. 
      <a href="?post=vode">vode</a>. <a href="#">#css</a>
      <a href="#">#responsive</a>
      <br />
      <time datetime="2025-09-24">10:09 PM - 24 Sep 2025</time>
    </div>
  </div>
</div>
```

expressed as *vode* it would look like this:

```typescript
[DIV, { class: 'card' },
    [DIV, { class: 'card-image' },
        [FIGURE, { class: 'image is-4by3' },
            [IMG, {
                src: 'placeholders/1280x960.png',
                alt: 'Placeholder image'
            }]
        ]
    ],
    [DIV, { class: 'card-content' },
        [DIV, { class: 'media' },
            [DIV, { class: 'media-left' },
                [FIGURE, { class: 'image is-48x48' },
                    [IMG, {
                        src: 'placeholders/96x96.png',
                        alt: 'Placeholder image'
                    }]
                ]
            ],
            [DIV, { class: 'media-content' },
                [P, { class: 'title is-4' }, 'John Smith'],
                [P, { class: 'subtitle is-6' }, '@johnsmith']
            ]
        ],
        [DIV, { class: 'content' },
            'Lorem ipsum dolor sit amet, consectetur adipiscing elit. ',
            [A, {href: '?post=vode'}, 'vode'], '. ', [A, { href: '#' }, '#css'],
            [A, { href: '#' }, '#responsive'],
            [BR],
            [TIME, { datetime: '2025-09-24' }, '10:09 PM - 24 Sep 2025']
        ]
    ]
]
```

Viewed in isolation, it does not provide an obvious benefit (apart from looking better imho), 
but as the result of a function of state, it can become very useful to express conditional UI this way. 

### component

```typescript
type Component<S> = (s: S) => ChildVode<S>;
```

A `Component<State>` is a function that takes a state object and returns a `Vode<State>` or `string`. 
It is used to render the UI based on the current state. 
A new *vode* must be created on each render, otherwise it would be skipped which could lead to unexpected results. If you seek to improve render performance, have a look at the [`memo`](#memoization) function.

```typescript
// A full vode has a tag, properties, and children. props and children are optional.
const CompFoo = (s) => [SPAN, { class: "foo" }, s.isAuthenticated ? "foo" : "bar"];

const CompBar = (s) => [DIV, { class: "container" }, 
    
    // a child vode can be a string, which results in a text node
    [H1, "Hello World"], 
    
    // a vode can also be a self-closing tag
    [HR],

    // conditional rendering
    s.isAuthenticated 
        ? [STRONG, `and also hello ${s.user}`]
        : [FORM,
            [INPUT, { type: "email", name: "email" }],
            [INPUT, { type: "password", name: "pw" }],
            [INPUT, { type: "submit" }],
        ],
    // a child-vode of false, undefined or null is not rendered 
    !s.isAuthenticated && [HR],
    
    // style object maps directly to the HTML style attribute
    [P, { style: { color: "red", fontWeight: "bold" } }, "This is a paragraph."],
    [P, { style: "color: red; font-weight: bold;" }, "This is also a paragraph."],

    // class property has multiple forms
    [UL,
        [LI, {class: "class1 class2"}, "as string"],
        [LI, {class: ["class1", "class2"]}, "as array"],
        [LI, {class: {class1: true, class2: false}}, "as Record<string, boolean>"],
    ],

    // events get the state object as first argument
    // and the HTML event object as second argument
    [BUTTON, {
        // all on* events accept `Patch<State>`
        onclick: (s, evt) => {
            // objects returned by events are patched automatically
            return { counter: s.counter + 1 }; 
        },

        // you can set the patch object directly for events
        onmouseenter: { pointing: true },
        onmouseleave: { pointing: false },

        // a patch can be an async function
        onmouseup: async (s, evt) => {
            s.patch({ loading: true });
            const result = await apiCall();
            return { title: result.data.title, loading: false };
        },

        // you can also use a generator function that yields patches
        onmousedown: async function* (s, evt) {
            yield { loading: true }; 
            const result = await apiCall();
            yield { 
                body: result.data.body,
            };
            return { loading: false };
        },

        // events can be attached conditionally
        ondblclick : s.counter > 20 && (s, evt) => {
            return { counter: s.counter * 2 }; 
        },

        class: { bar: s.pointing }
    }, "Click me!"],

    // components can be used as child-vodes, they are called lazy on render
    CompFoo,
    // or this way
    CompFoo(s),
];
```

### app

`app` is a function that takes a HTML node, a state object, and a render function (`Component<State>`).  

```typescript
const containerNode = document.getElementById('ANY-ELEMENT');
const state = {
    counter: 0,
    pointing: false,
    loading: false,
    title: '',
    body: '',
};

const patch = app(
    containerNode, 
    state, 
    (s) => 
        [DIV, 
            [P, { style: { color: 'red' } }, `${s.counter}`],
            [BUTTON, { onclick: () => ({ counter: s.counter + 1 }) }, 'Click me'],    
        ]
    );
```

It will analyze the current structure of the given `ContainerNode` and adjust its structure in the first render. 
When render-patches are applied to the `patch` function or via yield/return of events, 
the `ContainerNode` is updated to match the vode structure 1:1. 

#### defuse

To release resources associated with the vode app instance, you can call the `defuse` function on the `ContainerNode` that was passed to `app`.

```typescript
import { app, defuse } from '@ryupold/vode';
const containerNode = document.getElementById('ANY-ELEMENT');
const state = { /* ... */ };
app(containerNode, state, s => /* ... */ );
//... later ...
// when you want to clean up the vode app instance
defuse(containerNode);
```

The DOM elements created by the vode app will remain in the `ContainerNode`, but all event listeners and references to the state object will be removed, allowing for proper garbage collection.

### state & patch
The state object you pass to [`app`](#app) can be updated directly or via `patch`. 
During the call to `app`, the state object is bound to the vode app instance and becomes a singleton from its perspective. 
Also a `patch` function is added to the state object; it is the same function that is also returned by `app`.
A re-render happens when a patch object is supplied to the `patch` function or via event.
When an object is passed to `patch`, its properties are recursively deep merged onto the state object.

```js
const s = {
    counter: 0,
    pointing: false,
    loading: false,
    title: 'foo',
    body: '',
};

app(appNode, s, s => AppView(s)); 
// after calling app(), the state object is bound to the appNode

// update state directly as it is a singleton (silent patch, no render)
s.title = 'Hello World';

// render patch
s.patch({});

// render patch with a change that is applied to the state 
s.patch({ title: 'bar' }); 

// patch with a function that receives the state
s.patch((s) => ({body: s.body + ' baz'})); 

// patch with an async function that receives the state
s.patch(async (s) => {
    s.loading = true; // sometimes it is easier to combine a silent patch
    s.patch({});      // with an empty render patch
    const result = await apiCall();
    return { title: result.title, body: result.body, loading: false };
}); 

// patch with an async generator function that yields patches
s.patch(async function*(s){
    yield { loading: true };
    const result = await apiCall();
    yield { title: result.title, body: result.body };
    return { loading: false }; 
});

// ignored, also: undefined, number, string, boolean, void
s.patch(null);

// setting a property in a patch to undefined deletes it from the state object
s.patch({ pointing: undefined });

// ❌ it is discouraged to patch inside the render step 💩
const ComponentEwww = (s) => {
    if(!s.isLoading)
        s.patch(() => startLoading());

    return [DIV, s.isLoading ? [PROGRESS] : s.title];
}

// ✨ experimental view transitions support ✨
// patch with a render via view transition
s.patch([{}, (s) => {/*...*/}]); //all given patches will be part of a view transition

// empty array patches command to skip the current view transition
// and set the queued animated patches until now as current state with a sync patch
s.patch([]);

// skip current view transition and start this view transition instead
s.patch([[], { loading: true }]);
```

### memoization
To optimize performance, you can use `memo(depsArray, Component | PropsFactory)` to cache the result of a component function. If the array of dependencies does not change (shallow compare), the component function is not called again, indicating for the render to skip this node and all its children.
This is useful when the creation of the vode is expensive or the rendering of it takes a significant amount of time.


```typescript
const CompMemoList = (s) => 
    [DIV, { class: "container" }, 
        [H1, "Hello World"], 
        [BR], 
        [P, "This is a paragraph."],
        
        // expensive component to render
        memo(
            // dependency array is shallow compared (using === operator) to the previous renders' memo dependencies
            [s.title, s.body], 
            // this is the component function that will be 
            // called only when the array changes
            (s) => {
                const list = [UL];
                for (let i = 0; i < 1000; i++) {
                    list.push([LI, `Item ${i}`]);
                }
                return list;
            },
        )
    ];
```
Passing an empty dependency array means the component is only rendered once and then ignored.

You can also pass a function that returns the Props object to memoize the attributes.

```typescript
const CompMemoProps = (s) => [DIV, 
    memo([s.isActive], (s) => ({ 
        class: s.isActive ? 'active' : 'inactive' 
    })),
    "Content"
];
```

### error handling

You can catch errors during rendering by providing a `catch` property in the vode props.

```typescript
const CompWithError: ChildVode = () =>
    [DIV,
        {
            catch: (s: unknown, err: any) => [SPAN, { style: { color: 'red' } }, `An error occurred: ${err?.message}`],
        },

        [P, "Below error is intentional for testing error boundaries:"],

        [DIV, {
            // catch: [SPAN, { style: { color: 'red' } }, `An error occurred!`], // uncomment to catch child error directly here
            onMount: () => {
                throw new Error("Test error boundary in post view....");
            }
        }],
    ];
```

If the `catch` property is a function, it will be called with the current state and the error as arguments, and should return a valid child-vode to render instead.
If it is a vode, it will be rendered directly. 
If no `catch` property is provided, the error will propagate to the nearest ancestor that has a `catch` property defined, or to the top-level app if none is found.
Try to keep the `catch` blocks as specific as possible to avoid masking other errors. 
Or just don't make errors happen in the first place :)

### helper functions

The library provides some helper functions for common tasks.

```typescript
import { tag, props, children, mergeClass, hydrate } from '@ryupold/vode';

// Merge class props intelligently
mergeClass('foo', ['baz', 'bar']);  // -> 'foo baz bar'
mergeClass(['foo'], { bar: true, baz: false }); // -> 'foo bar'
mergeClass({zig: true, zag: false}, 'foo', ['baz', 'bar']);  // -> 'zig foo baz bar'

// Merge style props intelligently
mergeStyle({ color: 'red' }, 'font-weight: bold;'); // -> 'color: red; font-weight: bold;'
mergeStyle('color: white; background-color: blue;', { marginTop: '10px', color: 'green' }); // -> 'background-color: blue; margin-top: 10px; color: green;'

const myVode = [DIV, { class: 'foo' }, [SPAN, 'hello'], [STRONG, 'world']];

// access parts of a vode
tag(myVode);        // 'div'
props(myVode);      // { class: 'foo' }
children(myVode);   // [[SPAN, 'hello'], [STRONG, 'world']]

// get existing DOM element as a vode (can be helpful for analyzing/debugging)
const asVode = hydrate(document.getElementById('my-element'));
```

Additionally to the standard HTML attributes, you can define 2 special event attributes: 
`onMount(State, Element)` and `onUnmount(State, Element)` in the vode props. 
These are called when the element is created or removed during rendering. 
They receive the `State` as the first argument and the DOM element as the second argument.
Like the other events they can be patches too. 
> Be aware that `onMount/onUnmount` are only called when an element 
> is actually created/removed which might not always be the case during 
> rendering, as only a diff of the virtual DOM is applied.

### SVG & MathML
SVG and MathML elements are supported but need the namespace defined in properties.

```typescript
import { SVG, CIRCLE } from '@ryupold/vode';

const CompSVG = (s) => 
    [SVG, { xmlns: 'http://www.w3.org/2000/svg', width: 100, height: 100 },
        [CIRCLE, { cx: 50, cy: 50, r: 40, stroke: 'green', 'stroke-width': 4, fill: 'yellow' }]
    ];
```

```typescript
import { MATH, MSUP, MI, MN } from '@ryupold/vode';

const CompMathML = (s) => 
    [MATH, { xmlns: 'http://www.w3.org/1998/Math/MathML' },
        [MSUP, 
            [MI, 'x'], 
            [MN, '2']
        ]
    ];
```

### advanced usage

#### state context

The state context utilities can help creating shareable type safe components.

```typescript

type Settings = { theme: string, lang: string };
type StateType = {
    user: {
        profile: { settings: Settings }
    }
};

const state = createState<StateType>({
    user: {
        profile: {
            settings: { theme: 'dark', lang: 'en' }
        }
    }
});

// Create a context for the nested settings
const settingsCtx = new KeyStateContext<typeof state, Settings>(state, 'user.profile.settings');

const element = document.getElementById('app')!;
app(element, state,
    (s) => [DIV,
        [H1, "Settings"],
        SettingsForm(settingsCtx),
    ]
);

function SettingsForm(ctx: SubStateContext<Settings>) {
    const settings = ctx.get()!; // { theme: 'dark', lang: 'en' }

    return <Vode>[FORM,
        [P, "current theme:", settings.theme],
        [SELECT,
            {
                class: 'theme-select',
                onchange: (s: PatchableState, e: Event) => ctx.patch({ theme: (<HTMLSelectElement>e.target).value }),
            },
            [OPTION, { value: 'light', selected: settings.theme === 'light' ? '' : null }, 'light'],
            [OPTION, { value: 'dark', selected: settings.theme === 'dark' ? '' : null }, 'dark'],
        ],
        [P, "current lang:", settings.lang],
        [SELECT, {
            class: 'lang-select',
            onchange: (s: PatchableState, e: Event) => ctx.patch({ lang: (<HTMLSelectElement>e.target).value }),
        },
            [OPTION, { value: 'en', selected: settings.lang === 'en' ? '' : null }, 'en'],
            [OPTION, { value: 'de', selected: settings.lang === 'de' ? '' : null }, 'de'],
            [OPTION, { value: 'es', selected: settings.lang === 'es' ? '' : null }, 'es'],
            [OPTION, { value: 'fr', selected: settings.lang === 'fr' ? '' : null }, 'fr'],
        ],
    ];
}
```

#### isolated state
You can have multiple isolated vode app instances on a page, each with its own state and render function.
The returned patch function from `app` can be used to synchronize the state between them.

#### nested vode-app
It is possible to nest vode-apps inside vode-apps, but the library is not opinionated on how you do that. 
One can imagine this type of component:

```typescript
export function IsolatedVodeApp<OuterState, InnerState>(
    tag: Tag,
    state: InnerState,
    View: (ins: InnerState) => Vode<InnerState>,
): ChildVode<OuterState> {
    return memo<OuterState>([],
        () => [tag,
            {
                onMount: (s: OuterState, container: Element) => {
                    app<InnerState>(container, state, View);
                }
            }
        ]
    );
}
```
The memo with empty dependency array prevents further render calls from the outer app
so rendering of the subtree inside is controlled by the inner app.
Take note of the fact that the top-level element of the inner app refers to the surrounding element and will change its state accordingly.

#### view transitions
The library has experimental support for [View Transitions API](https://developer.mozilla.org/en-US/docs/Web/API/View_Transition_API).
You can pass an array of patches to the `patch` function where each patch will be applied with the next available view transition.

Patching an empty array `[]` will skip the current view transition and set the queued animated patches until now as current state with a sync patch. 
> Keep in mind that view transitions are not supported in all browsers yet and only one active transition can happen at a time. This feature may change significantly in the future, so do not rely on it heavily.

Scheduling behaviour can in theory be overridden with `containerNode._vode.asyncRenderer`.

```js
// or globally disable view transitions for the vode framework
import { globals } from '@ryupold/vode';
globals.startViewTransition = null;

// set to disable view transitions for specific vode-app
containerNode._vode.asyncRenderer = null;
```

### performance

There are some metrics available on the appNode. 
They are updated on each render.

```typescript
app<State>(appNode, state, (s) => ...);

console.log(appNode._vode.stats);
```

```javascript
{
    // number of patches applied to the state overall
    patchCount: 100,
    // number of render-patches (objects) overall
    syncRenderPatchCount: 55,
    // number of view transition render-patches (arrays) overall
    asyncRenderPatchCount: 3,
    // number of sync "normal" renders performed overall
    syncRenderCount: 43,
    // number of async renders performed overall
    asyncRenderCount: 2,
    // time the last render took in milliseconds
    lastSyncRenderTime: 2,
    // time the last view transition took in milliseconds
    lastAsyncRenderTime: 21,
    // number of active async running effects (function based patches)
    liveEffectCount: 0,
}
```

The library is optimized for small to medium sized applications. In my own tests it could easily handle sites with tens of thousands of elements. Smart usage of `memo` can help to optimize performance further. You can find a comparison of the performance with other libraries [here](https://krausest.github.io/js-framework-benchmark/current.html).

This being said, the library does not focus on performance.
It is designed to feel nice while coding, by providing a primitive that is simple to bend & form.
I want the mental model to be easy to grasp and the API surface to be small 
so that a developer can focus on building a web application instead of learning the framework and get to a flow state as quick as possible.

## Thanks

The simplicity of [hyperapp](https://github.com/jorgebucaran/hyperapp) demonstrated that powerful frameworks don't require complexity, which inspired this library's design philosophy.

Not planning to add more features, just keeping it simple and easy (and hopefully bug free).

But if you find bugs or have suggestions, 
feel free to open an [issue](https://github.com/ryupold/vode/issues) or a pull request.

## License
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](./LICENSE)